
-- A bus controller for multiple processes 
-- The bus controller communicates with the PC via an ISA send/rec channel
-- and with the processes via CL_DXP2 send/rec channels

-- Assumptions :
-- 1. The '-', '*' and mod operations for STD_LOGIC vectors is supported by
--    the high level synthesis tool or operator overlaod functions have
--    been written.
-- 2. The testbit function has been written separately.

library IEEE;
use IEEE.STD_LOGIC_1164.all;

use work.cl_types.all;
use work.CL_ISA.all;
use work.CL_DXPX.all;

-- port definition
entity CL_S_ISA is
       generic(N: INTEGER := 8);
       port(--The following signals are common to both input and output
            addr_p               : in bit16;
            aen_p                : in bit;
            rst_p                : in bit;
            ioch_rdy_p           : out bit;
            clk_p                : in bit;
            data_p               : inout STD_LOGIC8;  
            -- Input channels only
            iow_p                : in bit; 
            -- Output channels only
            ior_p                : in bit);  
end CL_S_ISA;

architecture ACL_S_ISA of CL_S_ISA is 

-- The following ISA addresses are from the unused block 0300h-0377h i.e.
-- 0301h, 0302h, 0303h and 0304h

-- The first 12 MSB bits are common and are used to identify the module,   
-- zero extended to 16 for uniformity 
constant ISA_module_addr         : bit16 := "0000001100000000";

-- The next 4 bits are used to identify the various send and receive
-- addresses for the processes 
constant gcd_send_addr           : bit4 := "0000";
constant gcd_rec_addr            : bit4 := "0001";
constant rsa_send_addr           : bit4 := "0010";
constant rsa_rec_addr            : bit4 := "0011";

-- The CL_D8P2 bus is used for communication between the bus
-- controller and the processes. Each process has a disitinct request
-- line instead of using the addressed mode. Non addressed mode is
-- faster and only a reasonable amount of processes can be assumed to
-- run simultaneously on the FPGA 
signal bc_data_s                 : STD_LOGIC8;  
signal bc_gcd_req_s              : STD_LOGIC;
signal bc_rsa_req_s              : STD_LOGIC;

-- The global signals shared between the processes and the bus controllers
-- to aid in sending the sentinel 
signal gcd_done_s                : bit := '0';
signal rsa_done_s                : bit := '0';

-- The send and receive channels for the different processes for
-- the D8P2 interface
signal gcd_send_chan_D8P2_s      : CL_SEND_S_D8P2;
signal gcd_rec_chan_D8P2_s       : CL_REC_S_D8P2;
signal rsa_send_chan_D8P2_s      : CL_SEND_S_D8P2;
signal rsa_rec_chan_D8P2_s       : CL_REC_S_D8P2;

-- The send and receive channels for the bus controller for the D8P2 interface
signal gcd_mst_send_chan_D8P2_s  : CL_SEND_M_D8P2;
signal gcd_mst_rec_chan_D8P2_s   : CL_REC_M_D8P2;
signal rsa_mst_send_chan_D8P2_s  : CL_SEND_M_D8P2;
signal rsa_mst_rec_chan_D8P2_s   : CL_REC_M_D8P2;

-- The send and receive channele for the bus controller for the ISA interface
signal send_chan_ISA_s           : CL_SEND_S_ISA;
signal rec_chan_ISA_s            : CL_REC_S_ISA;

-- Use by the RSA process to encode a character
function EncodeMsg(val_a:STD_LOGIC8; val_b:STD_LOGIC32; val_n:STD_LOGIC32) 
                                                         return STD_LOGIC32 is
variable val_d            : STD_LOGIC32 := "00000000000000000000000000000001";
begin
   for i in 31 to 0 loop
      val_d := (val_d * val_d) mod val_n;
      if testbit(val_b, i) then
         val_d := (val_d * val_a) mod val_n;
      end if;
   end loop;

   return val_d;
end EncodeMsg;

-- The following routine to receive a long builds upon the routines
-- in the CL_D8P2 library
procedure CL_RecLongSrvD8P2(
            signal   chan              : in CL_REC_S_D8P2;
            variable rec_long          : out STD_LOGIC32;   
            signal   data_p            : in STD_LOGIC8;
            signal   data_req_p        : in STD_LOGIC;
            signal   clk_p             : in bit) is
begin
   -- Receive the higher order bytes followed by the lower order
   CL_RecSrvD8P2(chan, rec_long(31 downto 24), data_p, data_req_p, clk_p);
   CL_RecSrvD8P2(chan, rec_long(23 downto 16), data_p, data_req_p, clk_p);
   CL_RecSrvD8P2(chan, rec_long(15 downto 8), data_p, data_req_p, clk_p);
   CL_RecSrvD8P2(chan, rec_long(7  downto 0), data_p, data_req_p, clk_p);
end CL_RecLongSrvD8P2;

-- The following routine to send a long builds upon the routines
-- in the CL_D8P2 library
procedure CL_SendLongSrvD8P2(
            signal   chan              : in CL_SEND_S_D8P2;
            constant send_long         : in STD_LOGIC32;   
            signal   data_p            : out STD_LOGIC8;
            signal   data_req_p        : in STD_LOGIC;
            signal   clk_p             : in bit) is
begin
   -- Send the higher order bytes followed by the lower order
   CL_SendSrvD8P2(chan, send_long(31 downto 24), data_p, data_req_p, clk_p);
   CL_SendSrvD8P2(chan, send_long(31 downto 24), data_p, data_req_p, clk_p);
   CL_SendSrvD8P2(chan, send_long(31 downto 24), data_p, data_req_p, clk_p);
   CL_SendSrvD8P2(chan, send_long(31 downto 24), data_p, data_req_p, clk_p);
end CL_SendLongSrvD8P2;

begin
   bus_controller:process
      variable ISA_Ready         : bit  := '0';
      variable ISA_module_addr_lsb : int4 := 4;
      variable read_addr         : bit16;
      variable char_data         : STD_LOGIC8;
   begin
      -- Initialize the ISA channels
      CL_InitSendSrvISA(send_chan_ISA_s, data_p, ISA_module_addr, clk_p);   
      CL_InitRecSrvISA(rec_chan_ISA_s, data_p, ISA_module_addr, clk_p);
    
      loop
         while(ISA_Ready = '0') loop      
            CL_ReadySendSrvISA(send_chan_ISA_s, ISA_Ready, ISA_module_addr_lsb,
                   read_addr, addr_p, aen_p, rst_p, ior_p, ioch_rdy_p, clk_p);
            CL_ReadyRecSrvISA(rec_chan_ISA_s, ISA_Ready, ISA_module_addr_lsb, 
                   read_addr, addr_p, aen_p, rst_p, iow_p, ioch_rdy_p, clk_p);
         end loop;
         
         case read_addr(3 downto 0) is
            when gcd_send_addr =>
               CL_GetDataSrvISA(rec_chan_ISA_s, char_data, data_p, 
                                                    ioch_rdy_p,iow_p, clk_p);
               CL_SendMstD8P2(gcd_mst_send_chan_D8P2_s, char_data, bc_data_s,
                                                          bc_gcd_req_s,clk_p);
            when gcd_rec_addr  =>
               if (gcd_done_s = '0') then
                  CL_PutDataSrvISA(send_chan_ISA_s, "00000000", data_p,
                                                    ioch_rdy_p, ior_p, clk_p);
               else
                  CL_RecMstD8P2(gcd_mst_rec_chan_D8P2_s, char_data, bc_data_s,
                                                          bc_gcd_req_s,clk_p);

                  CL_PutDataSrvISA(send_chan_ISA_s, char_data, data_p,
                                                    ioch_rdy_p, ior_p, clk_p);
               end if;

            when rsa_send_addr =>
               CL_GetDataSrvISA(rec_chan_ISA_s, char_data, data_p, 
                                                    ioch_rdy_p, iow_p, clk_p);
               CL_SendMstD8P2(rsa_mst_send_chan_D8P2_s, char_data, bc_data_s,
                                                          bc_gcd_req_s, clk_p);
            when rsa_rec_addr  =>
               if (rsa_done_s = '0') then
                  CL_PutDataSrvISA(send_chan_ISA_s, "00000000", data_p,
                                                    ioch_rdy_p, ior_p, clk_p);
               else
                  CL_RecMstD8P2(rsa_mst_rec_chan_D8P2_s, char_data, bc_data_s,
                                                          bc_gcd_req_s, clk_p);

                  CL_PutDataSrvISA(send_chan_ISA_s, char_data, data_p,
                                                    ioch_rdy_p, ior_p, clk_p);
               end if;
            when others =>
               null;                          
         end case;   
      end loop;
   end process;

   gcd:process
      variable x                 : STD_LOGIC8;
      variable y	         : STD_LOGIC8;     
   begin   
      -- Initializations
      CL_InitSendSrvD8P2(gcd_send_chan_D8P2_s, bc_data_s, bc_gcd_req_s, clk_p);
      CL_InitRecSrvD8P2(gcd_rec_chan_D8P2_s, bc_data_s, bc_gcd_req_s, clk_p);

      loop
         gcd_done_s  <= '0';

         -- Receive x    
         CL_RecSrvD8P2(gcd_rec_chan_D8P2_s, x, bc_data_s, bc_gcd_req_s, clk_p);

         -- Receive y
         CL_RecSrvD8P2(gcd_rec_chan_D8P2_s, y, bc_data_s, bc_gcd_req_s, clk_p);

         -- Compute the gcd 
         while (x /= y) loop
            if(x < y) then
               y := y-x;
            else x:= x-y;
            end if;
         end loop;
    
         -- Set the gcd global  
         gcd_done_s  <= '1';
         wait until clk_p'event and clk_p='1';
               
         -- Send the result
         CL_SendSrvD8P2(gcd_send_chan_D8P2_s, x, bc_data_s, bc_gcd_req_s, 
                                                                       clk_p); 
      end loop;
   end process;

   rsa:process
      variable  pubkey_d         : STD_LOGIC32;
      variable  pubkey_n         : STD_LOGIC32;     
      variable  msg_item_raw     : STD_LOGIC8;
      variable  msg_item_encoded : STD_LOGIC32;

   begin 
      -- Initializations
      CL_InitSendSrvD8P2(rsa_send_chan_D8P2_s, bc_data_s, bc_rsa_req_s, clk_p);
      CL_InitRecSrvD8P2(rsa_rec_chan_D8P2_s, bc_data_s, bc_rsa_req_s, clk_p);

      -- Receive the public keys
      CL_RecLongSrvD8P2(rsa_rec_chan_D8P2_s, pubkey_d, bc_data_s, 
                                                          bc_rsa_req_s, clk_p);
      CL_RecLongSrvD8P2(rsa_rec_chan_D8P2_s, pubkey_n, bc_data_s, 
                                                          bc_rsa_req_s, clk_p);

      loop
         rsa_done_s <= '0'; 

         -- Receive a character
         CL_RecSrvD8P2(rsa_rec_chan_D8P2_s, msg_item_raw, bc_data_s, 
                                                          bc_rsa_req_s, clk_p);
         -- Encode the message
         msg_item_encoded := EncodeMsg(msg_item_raw, pubkey_d, pubkey_n);
  
         -- Set the RSA global
         rsa_done_s <= '1'; 
   
         -- Send the result
         CL_SendLongSrvD8P2(rsa_send_chan_D8P2_s, msg_item_encoded, bc_data_s, 
                                                          bc_rsa_req_s, clk_p);
      end loop; 
   end process; 
   
end ACL_S_ISA;
