-- FPGA protocol for the ISA bus interface (8 bits) 
-- Flag address approach
-- Assumptions :
-- 1. The "-" operator has been overlaoded for bit vectors.
-- 2. An assignment operation can be done from normal to tristate variable 

use work.CL_ISA.all;

-- port definition
entity CL_S_ISA is
       generic(N: INTEGER := 8);
       port(-- The following signals are common to both input and output
            addr_p               : in bit16;
            aen_p                : in bit;
            rst_p                : in bit;
            clk_p                : in bit; 
            -- Input channels only
            iow_p                : in bit;
            datain_p             : in bit8;    
            -- Output channels only
            ior_p                : in bit;
            dataout_p            : out tristate_bit8);  
end CL_S_ISA;

architecture ACL_S_ISA of CL_S_ISA is 

-- Using addresses from the unused block 0300h-0377h
constant gcd_send_addr           : bit16 := "0000001100000000";  
constant gcd_rec_addr            : bit16 := "0000001100000001";
constant gcd_done_addr           : bit16 := "0000001100000010";

signal gcd_done                  : bit := '0';

-- Declare the send and receive channels
signal send_chan                 : CL_SEND_S_ISA;
signal rec_chan                  : CL_REC_S_ISA;
signal gcd_done_chan             : CL_SEND_S_ISA;

begin
   gcd:process
      variable x                 : bit8;
      variable y	         : bit8;
      variable result            : tristate_bit8;       
   begin   
      -- Initializations
      CL_InitSendSrvISA(send_chan,dataout_p,gcd_send_addr,clk_p);
      CL_InitRecSrvISA(rec_chan,gcd_rec_addr,clk_p);

      loop
         gcd_done <= '0';

         -- Receive x    
         CL_RecSrvISA(rec_chan,x,addr_p,datain_p,aen_p,rst_p,iow_p,clk_p);

         -- Receive y
         CL_RecSrvISA(rec_chan,y,addr_p,datain_p,aen_p,rst_p,iow_p,clk_p);

         -- Compute the gcd 
         while (x /= y) loop
            if(x < y) then
               y := y-x;
            else x:= x-y;
            end if;
         end loop;
    
         result := x;

         -- Set the gcd flag  
         gcd_done  <= '1';
            
         -- vcom warns of infinite loop without a wait in process, inlining
         -- should mean that the following wait is not needed
         wait until clk_p'event and clk_p='1';
               
         -- Send the result
         CL_SendSrvISA(send_chan, result, addr_p, dataout_p, aen_p, rst_p,
                                                             ior_p, clk_p);
      end loop;
   end process;

   flag:process
   begin
      -- Initialization   
      CL_InitSendSrvISA(gcd_done_chan,dataout_p,gcd_done_addr,clk_p);
 
      loop

         -- Tell the ISA master that the gcd result is not computed yet 

         while (gcd_done = '0') loop
            CL_SendSrvISA(gcd_done_chan, "00000000", addr_p, dataout_p, aen_p,
                                                       rst_p, ior_p, clk_p);  
         end loop;

         -- gcd done 
         CL_SendSrvISA(gcd_done_chan, "00000001", addr_p, dataout_p, aen_p,
                                                        rst_p, ior_p, clk_p);
        
         wait until clk_p'event and clk_p='1' and gcd_done='0';
      end loop;
   end process;

end ACL_S_ISA;
