--------------------------------------------------------------------------------
--  Title:     Memory Package
--
--  Created:   Thu Apr 15 11:54:50 1999
--  Author:    Vijay A. Nebhrajani	<vijayn@himgiri.comit.com>
--  Source File Name:   mempkg.vhd
--
--  $Id: mempkg.vhd,v 1.1 1999/04/15 11:54:50 vijayn Exp $
--
--  Description:   This is a VHDL 93 file.
--
--  Revision History:   
--
--  $Log: mempkg.vhd,v $
--------------------------------------------------------------------------------

library IEEE, STD, WORK;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;

package mempkg is
  shared variable MEM_DEPTH : natural;   --  Address bus width
  shared variable MEM_WIDTH : natural;   --  Data bus width

  type vmem;

  type slv_ptr is access std_logic_vector;

  type vmem_ptr is access vmem;

  --  Basic memory element - each is a linked list element.
  type vmem is
    record
      address             : slv_ptr;
      data                : slv_ptr;
      nextp               : vmem_ptr;
      prevp               : vmem_ptr;
    end record;

  impure function create_mem
    (depth      : natural; 
     width      : natural
    ) 
    return vmem_ptr;
  
  procedure write_mem
    (variable mem_ptr     : in  vmem_ptr;
     address              : in  std_logic_vector(MEM_DEPTH - 1 downto 0);
     data                 : in  std_logic_vector(MEM_WIDTH - 1 downto 0)
    );
  
  procedure read_mem
    (variable mem_ptr     : in  vmem_ptr;
     address              : in  std_logic_vector(MEM_DEPTH - 1 downto 0);
     signal data          : out std_logic_vector(MEM_WIDTH - 1 downto 0)
    );
  
end mempkg;

package body mempkg is
  
  -------------------------------------------------------------------------
  --  Function:          create_mem()
  --  Description:       Creates a new memory structure, loads up the
  --                     global MEM_DEPTH and MEM_WIDTH variables and
  --                     returns a pointer to the memory structure.
  -------------------------------------------------------------------------
  impure function create_mem
    (depth      : natural; 
     width      : natural) 
    return vmem_ptr is
    
    variable new_memory_ptr : vmem_ptr;
  begin

    MEM_WIDTH := width;
    MEM_DEPTH := depth;
    new_memory_ptr := new vmem;
    new_memory_ptr.address := new std_logic_vector(MEM_DEPTH - 1 downto 0);
    new_memory_ptr.data    := new std_logic_vector(MEM_WIDTH - 1 downto 0);
    new_memory_ptr.nextp := NULL;
    new_memory_ptr.prevp := NULL;
    return new_memory_ptr;

  end create_mem;
  
  -------------------------------------------------------------------------
  --  Function:          write_mem()
  --  Description:       Writes a new data word to the address specified
  --                     in the memory structure. This writing is done
  --                     in an unsorted fashion, and in FIFO order. This
  --                     ensures that oldest writes will be accessed
  --                     faster in real time. This is a simulation time
  --                     issue, and for the modeler, it is a transparent
  --                     processes.
  -------------------------------------------------------------------------
  procedure write_mem
    (variable mem_ptr     : in  vmem_ptr;
     address              : in  std_logic_vector(MEM_DEPTH - 1 downto 0);
     data                 : in  std_logic_vector(MEM_WIDTH - 1 downto 0)
    ) is
    
    variable current_ptr  : vmem_ptr;
    variable prev_ptr     : vmem_ptr;
    variable new_mem      : vmem_ptr;
    
  begin
    if (mem_ptr = NULL) then
      assert false
      report "Bad Memory Pointer, mem_ptr = NULL"
      severity failure;
    end if;
    --  The head is always empty :-)
    --  Scan to the end of the list.
    current_ptr := mem_ptr;
    while (current_ptr.nextp /= NULL) loop
      current_ptr := current_ptr.nextp;
    end loop;
    prev_ptr := current_ptr;
    new_mem := create_mem(MEM_DEPTH, MEM_WIDTH);
    current_ptr.nextp := new_mem;
    current_ptr := current_ptr.nextp;
    current_ptr.address.all := address;
    current_ptr.data.all    := data;
    current_ptr.nextp   := NULL;
    current_ptr.prevp   := prev_ptr;
  end write_mem;
  

  -------------------------------------------------------------------------
  --  Procedure:         read_mem()
  --  Description:       Reads a data word from the address specified. If
  --                     that address has not been written to, this
  --                     function returns X.
  -------------------------------------------------------------------------
  procedure read_mem
    (variable mem_ptr     : in  vmem_ptr;
     address              : in  std_logic_vector(MEM_DEPTH - 1 downto 0);
     signal data          : out std_logic_vector(MEM_WIDTH - 1 downto 0)
    ) is

    variable current_address : std_logic_vector(MEM_DEPTH - 1 downto 0);
    variable current_ptr  : vmem_ptr;
    
  begin
    current_address := address;
    current_ptr := mem_ptr;
    --  Scan the list. We are at the head.
    if (current_ptr = NULL) then
      --  Null memory element.
      data <= (others => 'X');
      return;
    end if;
    while (current_address /= current_ptr.address.all) loop
      if (current_ptr.nextp /= NULL) then
        current_ptr := current_ptr.nextp;
      else
        --  End of list reached, no address match.
        data <= (others => 'X');
        return;
      end if;
    end loop;
    data <= current_ptr.data.all;
  
  end read_mem;

end mempkg;
